/** @file   polygonobject.cpp
 * @brief   Implementation of the PolygonObject - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "polygonobject.h" // class's header file
#include "www_map.h"
#include "www_assert.h"
#include "redrawqueue.h"
#include <string>
using eng2d::Vec2D;
using std::string;
using std::istream;
using std::vector;

namespace WeWantWar {

///
/// Constructor, destructor
/// =======================


// class constructor
PolygonObject::PolygonObject() :
  GameObject()
{
	// insert your code here
}

/** Destructor.
 */
PolygonObject::~PolygonObject()
{
	for ( int i=0; i < this->polycount(); i++ ) {
    Polygon* pP = this->getPolygon( i );
    if ( pP != 0 ) {
      delete pP;
      pP = 0;
    }
  }
  m_polytable.clear();
}




///
/// Public methods
/// ==============

/** Redraws the polygon
 */
void PolygonObject::redraw( RedrawQueue* pQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  
  Vec2D tmpPos = this->position();
  
  for ( int i=0; i < this->polycount(); i++ ) {
    int polytype = POLYTYPE_FLAT;
    BITMAP* pTex = 0;
    
    
    Polygon* pP = this->getPolygon( i );
    
    // Get all the vertices of this polygon and rotate them.
    float minX = 640;
    float minY = 480;
    float maxX = 0;
    float maxY = 0;
    V3D_f* pVtx = new V3D_f[ pP->vertexCount() ];
    for (int j=0; j < pP->vertexCount(); j++ ) {
      pP->getRotatedVertex_ex( j, &pVtx[j], this->angle() );
      
      pVtx[j].x += tmpPos.x();
      pVtx[j].y += tmpPos.y();
      pVtx[j].x -= Map::scrollX;
      pVtx[j].y -= Map::scrollY;
      
      if ( pVtx[j].x < minX ) minX = pVtx[j].x;
      if ( pVtx[j].x > maxX ) maxX = pVtx[j].x;
      if ( pVtx[j].y < minY ) minY = pVtx[j].y;
      if ( pVtx[j].y > maxY ) maxY = pVtx[j].y;
    }
    
    if ( !(maxX < 0 || maxY < 0 || minX > 640 || minY > 480) ) {
      if ( pP->texture() >= 0 ) {
        // We use texture.
        pTex = this->getTexture( pP->texture() );
      }

      if ( pP->vertexCount() == 3 ) {
        pQueue->addTriangle( RedrawQueue::PRI_NORMAL, pP->polytype(),
                             pTex, &pVtx[0], &pVtx[1], &pVtx[2] );
      
      } else if ( pP->vertexCount() == 4 ) {
        pQueue->addQuad( RedrawQueue::PRI_NORMAL, pP->polytype(), pTex,
                         &pVtx[0], &pVtx[1], &pVtx[2], &pVtx[3] );
      
      }
    }
    delete [] pVtx;
  }
}



/** Adds new texture
 */
void PolygonObject::addTexture( BITMAP* pB )
{
  m_texturetable.push_back( pB );
}



/** Sets the texture
 */
void PolygonObject::setTexture( int index, BITMAP* pB )
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < m_texturetable.size() );
  
  m_texturetable.at(index) = pB;
}



/** Adds new polygon
 */
void PolygonObject::addPolygon( Polygon* pPoly )
{
  m_polytable.push_back( pPoly );
}



/** Reads the data from given input stream
 */
int PolygonObject::read( istream& rIn )
{
  string tmp;
  while (true) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    rIn >> tmp;
    if ( tmp == "</wewantwar_polygonobject>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "<polygon>" ) {
      int ret = this->readPolygon( rIn );
      if ( ret != 0 ) {
        return -1;
      }
    }
  }
  return 0;
}



///
/// Getter methods
/// ==============

/** Returns the pointer to texture bitmap
 */
BITMAP* PolygonObject::getTexture( int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < m_texturetable.size() );

  return m_texturetable.at( index );
}



/** Returns the number of polygons this object has.
 */
int PolygonObject::polycount() const
{
  return m_polytable.size();
}



/** Returns the polygon
 */
Polygon* PolygonObject::getPolygon( int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < m_polytable.size() );
  
  return m_polytable.at( index );
}



///
/// Protected methods
/// =================

/** Reads the polygon
 */
int PolygonObject::readPolygon( istream& rIn )
{
  vector< Polygon::Vtx > vertices;
  int tmpPolytype = 0;
  int tmpTexture = -1;
  
  int ptype_lookup[] = { POLYTYPE_FLAT, POLYTYPE_ATEX_MASK };
  
  string tmp;
  while ( true ) {
    rIn >> tmp;
    if ( tmp == "</polygon>" ) {
      // The polygon has been read. Now we create it and add it to this object.
      Polygon* pP = 0;
      if ( vertices.size() == 3 ) {
        pP = new Polygon( vertices.at(0), vertices.at(1), vertices.at(2) );
      } else if ( vertices.size() == 4 ) {
        pP = new Polygon( vertices.at(0), vertices.at(1),
                          vertices.at(2), vertices.at(3) );
      } else {
        return -1;
        
      }
      pP->polytype( ptype_lookup[tmpPolytype] );
      pP->texture( tmpTexture );
      
      this->addPolygon( pP );
      return 0;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "<vtx>" ) {
      // Read a vertex
      Polygon::Vtx tmpVtx;
      if ( this->readVertex( rIn, &tmpVtx ) != 0 ) {
        return -1;
      }
      vertices.push_back( tmpVtx );
      
    } else if ( tmp == "polytype:" ) {
      rIn >> tmpPolytype;
      
    } else if ( tmp == "texture:" ) {
      rIn >> tmpTexture;
      
    }
  }
}



/** Reads the vertex data
 */
int PolygonObject::readVertex( std::istream& rIn, Polygon::Vtx* pVtx )
{
  string tmp;
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "xy:" ) {
      float tmpX, tmpY;
      rIn >> tmpX >> tmpY;
      pVtx->coords = Vec2D( tmpX, tmpY );
      
    } else if ( tmp == "uv:" ) {
      float tmpU, tmpV;
      rIn >> tmpU >> tmpV;
      pVtx->tcoords = Vec2D( tmpU, tmpV );
      
    } else if ( tmp == "c:" ) {
      int r, g, b;
      rIn >> r >> g >> b;
      pVtx->c = eng2d::Color( r, g, b );
      
    } else if ( tmp == "</vtx>" ) {
      break;
    }
  }
  return 0;
}


} // end of namespace
